# API Documentation for WFM Dashboard Data

This document provides details on the API endpoints available for retrieving dashboard data from the WFM server at `172.20.60.110`. The APIs are served by three Flask applications:

- **ACD Dashboard API** (Port 5016): Provides agent and account data for ACD monitoring.
- **CW App API** (Port 5015): Provides queue and agent status data for call waiting dashboards.
- **QIS App API** (Port 5014): Provides QIS database data including agents, accounts, roles, statuses, user work info, and productivity reports.

All endpoints return JSON responses. Use HTTP GET requests to fetch data.

## Base URLs
- ACD Dashboard: `http://172.20.60.110:5016`
- CW App: `http://172.20.60.110:5015`
- QIS App: `http://172.20.60.110:5014`
- Realtime Dashboard: `http://172.20.60.110:5020`
- RT Hourly Mitel: `http://172.20.60.110:5005`

## Authentication
No authentication is required for these endpoints.

## Error Handling
- **200 OK**: Successful response with JSON data.
- **400 Bad Request**: Missing required parameters.
- **500 Internal Server Error**: Server-side error (check server logs).
- **504 Gateway Timeout**: Request timed out (CW App has a 25-second timeout).

---

## QIS App API (Port 5014)

### GET /api/agents
Retrieves a list of agents from the QIS cache.

**Request:**
```
GET http://172.20.60.110:5014/api/agents
```

**Response:**
```json
[
  {
    "UserID": "123",
    "UserName": "John Doe",
    "Name": "John Doe",
    "AccountID2": "ACC001",
    "IPPhoneID": "123",
    "Role": "Agent"
  }
]
```

**PHP Example:**
```php
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5014/api/agents');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$agents = json_decode($response, true);
```

### GET /api/accounts
Retrieves a list of accounts from the QIS database.

**Request:**
```
GET http://172.20.60.110:5014/api/accounts
```

**Response:**
```json
[
  {
    "AccountID": "ACC001",
    "AccountName": "Example Account"
  }
]
```

**PHP Example:**
```php
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5014/api/accounts');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$accounts = json_decode($response, true);
```

### GET /api/roles
Retrieves a list of roles from the QIS database.

**Request:**
```
GET http://172.20.60.110:5014/api/roles
```

**Response:**
```json
[
  "Agent",
  "Supervisor",
  "Manager"
]
```

**PHP Example:**
```php
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5014/api/roles');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$roles = json_decode($response, true);
```

### GET /api/statuses
Retrieves a list of statuses from the QIS database.

**Request:**
```
GET http://172.20.60.110:5014/api/statuses
```

**Response:**
```json
[
  "Active",
  "Inactive",
  "On Leave"
]
```

**PHP Example:**
```php
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5014/api/statuses');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$statuses = json_decode($response, true);
```

### GET /api/user-work-info
Retrieves user work information with productivity data.

**Request:**
```
GET http://172.20.60.110:5014/api/user-work-info?start_date=2025-11-01&end_date=2025-11-30&timezone=PH&aggregation_mode=default
```

**Parameters:**
- `start_date` (required): Start date in YYYY-MM-DD format.
- `end_date` (required): End date in YYYY-MM-DD format.
- `timezone` (optional): 'PH' or 'EST' (default: 'PH').
- `time_format` (optional): 'default' or 'seconds' (default: 'default').
- `aggregation_mode` (optional): 'default', 'aggregate', or 'aggregate_range' (default: 'default').
- `agent_id` (optional): Filter by specific agent ID.
- `account_id` (optional): Filter by specific account ID.
- `exclude_role` (optional, multiple): Roles to exclude.
- `exclude_status` (optional, multiple): Statuses to exclude.

**Response:**
```json
[
  {
    "UserID": 123,
    "Name": "John Doe",
    "WorkDate": "2025-11-01",
    "LocationName": "Office",
    "Branch": "Main",
    "RoleName": "Agent",
    "StatusName": "Active",
    "SchedTimeIn": "2025-11-01T09:00:00",
    "SchedTimeOut": "2025-11-01T17:00:00",
    "ActualTimeIn": "2025-11-01T09:05:00",
    "ActualTimeOut": "2025-11-01T17:15:00",
    "AccountID": "ACC001",
    "AccountName": "Example Account",
    "productivity_data": [
      {
        "shift_start_time": "2025-11-01T09:00:00",
        "shift_end_time": "2025-11-01T17:00:00",
        "total_shift_duration": 28800,
        "inbound_count": 50,
        "Inbound_handling_time": 7200,
        "outbound_count": 10,
        "outbound_time": 1800,
        "Occupancy_perc": 85.5,
        "Shrinkage_perc": 10.2,
        "AHT_In": 144,
        "AHT_Out": 180,
        "HoldRatio_perc": 5.0
      }
    ]
  }
]
```

**PHP Example:**
```php
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5014/api/user-work-info?start_date=2025-11-01&end_date=2025-11-30&timezone=PH');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$userWorkInfo = json_decode($response, true);
```

### GET /api/agent-productivity
Retrieves agent productivity report data.

**Request:**
```
GET http://172.20.60.110:5014/api/agent-productivity?start_date=2025-11-01&end_date=2025-11-30&timezone=PH
```

**Parameters:**
- `start_date` (required): Start date in YYYY-MM-DD format.
- `end_date` (required): End date in YYYY-MM-DD format.
- `timezone` (optional): 'PH' or 'EST' (default: 'PH').

**Response:**
```json
[
  {
    "agent_id": "123",
    "AgentName": "John Doe",
    "shift_date": "2025-11-01",
    "shift_start_time": "2025-11-01T09:00:00",
    "shift_end_time": "2025-11-01T17:00:00",
    "total_shift_duration": 28800,
    "Inbound_handling_time": 7200,
    "inbound_count": 50,
    "outbound_count": 10,
    "outbound_time": 1800,
    "Occupancy_perc": 85.5,
    "Shrinkage_perc": 10.2
  }
]
```

**PHP Example:**
```php
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5014/api/agent-productivity?start_date=2025-11-01&end_date=2025-11-30&timezone=PH');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$productivity = json_decode($response, true);
```

### POST /api/export-excel
Exports report data to Excel file.

**Request:**
```
POST http://172.20.60.110:5014/api/export-excel
Content-Type: application/json

{
  "report_data": [...],
  "selected_columns": ["Name", "WorkDate_formatted", "inbound_count"],
  "account_name": "AllAccounts",
  "timezone": "PH",
  "time_format": "default",
  "aggregation_mode": "default",
  "start_date": "2025-11-01",
  "end_date": "2025-11-30"
}
```

**Response:** Excel file download.

**PHP Example:**
```php
$data = [
    'report_data' => $reportData,
    'selected_columns' => ['Name', 'inbound_count'],
    'account_name' => 'AllAccounts',
    'timezone' => 'PH',
    'time_format' => 'default',
    'aggregation_mode' => 'default',
    'start_date' => '2025-11-01',
    'end_date' => '2025-11-30'
];

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5014/api/export-excel');
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);

// Handle file download
header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment; filename="export.xlsx"');
echo $response;
```

---

## Realtime Dashboard API (Port 5020)

### GET /api/realtime_data
Retrieves real-time agent productivity data for currently active agents.

**Request:**
```
GET http://172.20.60.110:5020/api/realtime_data
```

**Response:**
```json
[
  {
    "agent_id": "123",
    "AgentName": "John Doe",
    "shift_start_time": "2025-11-19T09:00:00",
    "shift_end_time": null,
    "total_shift_duration": 28800,
    "Inbound_handling_time": 7200,
    "inbound_count": 50,
    "answer_personal": 5,
    "inbound_talktime": 6500,
    "HoldTime": 700,
    "outbound_count": 10,
    "outbound_time": 1800,
    "ringingtime": 300,
    "make_busy_time": 1800,
    "work_timer": 1200,
    "water": 300,
    "Restroom": 600,
    "Break": 900,
    "Meal": 1800,
    "MeetingorTraining": 0,
    "ReviewingTickets": 0,
    "LongCaseNotes": 0,
    "SupervisorDuty": 0,
    "SpecialTasks": 0,
    "Other": 0,
    "nested_outbound_count": 2,
    "nested_outbound_time": 240,
    "HoldAbandonCount": 1,
    "HoldCount": 15,
    "call_avoidance": 0,
    "Occupancy_perc": 85.5,
    "Shrinkage_perc": 10.2
  }
]
```

**Notes:**
- Only returns data for agents with active shifts (shift_end_time is null or empty).
- Data is sorted alphabetically by agent name.
- All time fields are in seconds (integers).
- Percentage fields are rounded to 2 decimal places.

**PHP Example:**
```php
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5020/api/realtime_data');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$realtimeData = json_decode($response, true);
```

---

## RT Hourly Mitel API (Port 5005)

### POST /get_stats
Retrieves call statistics for a vendor on a specific date.

**Request:**
```
POST http://172.20.60.110:5005/get_stats
Content-Type: application/json

{
  "vendor_id": 100,
  "date": "2025-11-19"
}
```

**Response:**
```json
{
  "total": {
    "QueueName": "Vendor Name",
    "Offered": 150,
    "Handled": 140,
    "Abandoned": 10,
    "AvgSpeedofAnswer": "00:00:30",
    "AvgSpeedofAbandon": "00:01:15",
    "AvgHandlingTime": "00:05:20",
    "Abnrate": 6.67,
    "SLA": 85.5
  },
  "queues": [
    {
      "QueueName": "Queue1",
      "Offered": 50,
      "Handled": 48,
      "Abandoned": 2,
      "AvgSpeedofAnswer": "00:00:25",
      "AvgSpeedofAbandon": "00:01:00",
      "AvgHandlingTime": "00:04:30",
      "Abnrate": 4.0,
      "SLA": 90.0
    }
  ],
  "vendor_name": "Vendor Name"
}
```

**PHP Example:**
```php
$data = [
    'vendor_id' => 100,
    'date' => '2025-11-19'
];

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5005/get_stats');
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$stats = json_decode($response, true);
```

### POST /get_call_details
Retrieves detailed call information for a specific hour range.

**Request:**
```
POST http://172.20.60.110:5005/get_call_details
Content-Type: application/json

{
  "vendor_id": 100,
  "queue_name": "Queue1",
  "date": "2025-11-19",
  "hour_start": "09:00:00",
  "hour_end": "10:00:00"
}
```

**Response:**
```json
[
  {
    "CallType": "Answered",
    "QueueName": "Queue1",
    "EST_DateTime": "2025-11-19 09:15:30",
    "TotalHandlingTime": "00:04:25",
    "CallerWaitTime": "00:00:15",
    "ANI": "1234567890",
    "AgentID": "agent123"
  },
  {
    "CallType": "Abandoned",
    "QueueName": "Queue1",
    "EST_DateTime": "2025-11-19 09:30:45",
    "TotalHandlingTime": null,
    "CallerWaitTime": "00:02:10",
    "ANI": "0987654321",
    "AgentID": null
  }
]
```

**PHP Example:**
```php
$data = [
    'vendor_id' => 100,
    'queue_name' => 'Queue1',
    'date' => '2025-11-19',
    'hour_start' => '09:00:00',
    'hour_end' => '10:00:00'
];

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5005/get_call_details');
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$callDetails = json_decode($response, true);
```

### POST /get_hourly_stats
Retrieves hourly statistics breakdown for a vendor or specific queue.

**Request:**
```
POST http://172.20.60.110:5005/get_hourly_stats
Content-Type: application/json

{
  "vendor_id": 100,
  "date": "2025-11-19",
  "queue_name": "Queue1",
  "group_level": false
}
```

**Response:**
```json
{
  "hourly_stats": [
    {
      "Hour": "09:00",
      "Offered": 25,
      "Handled": 23,
      "Abandoned": 2,
      "AvgSpeedofAnswer": "00:00:20",
      "AvgSpeedofAbandon": "00:01:30",
      "AvgHandlingTime": "00:04:15",
      "Abnrate": 8.0,
      "SLA": 88.0
    }
  ],
  "queue_name": "Queue1",
  "is_group_level": false
}
```

**PHP Example:**
```php
$data = [
    'vendor_id' => 100,
    'date' => '2025-11-19',
    'queue_name' => 'Queue1',
    'group_level' => false
];

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5005/get_hourly_stats');
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$hourlyStats = json_decode($response, true);
```

---

## ACD Dashboard API (Port 5016)

### GET /api/accounts
Retrieves a list of all accounts available for filtering.

**Request:**
```
GET http://172.20.60.110:5016/api/accounts
```

**Response:**
```json
[
  {
    "account_id": "123",
    "account_name": "Example Account",
    "other_field": "value"
  }
]
```

**PHP Example:**
```php
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5016/api/accounts');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$accounts = json_decode($response, true);
```

### GET /api/agents
Retrieves detailed agent data, optionally filtered by account.

**Request:**
```
GET http://172.20.60.110:5016/api/agents
GET http://172.20.60.110:5016/api/agents?account=ExampleAccount
```

**Parameters:**
- `account` (optional): Filter agents by account name.

**Response:**
```json
[
  {
    "agent_id": "agent123",
    "agent_name": "John Doe",
    "account_name": "Example Account",
    "role": "Agent",
    "employee_id": "EMP001",
    "teamlead": "Jane Smith",
    "supervisor": "Bob Johnson",
    "accountmanager": "Alice Brown",
    "remote_worker": "No",
    "function_event": "Agent Answer Remote ACD",
    "reason_code": "",
    "makebusy_code": "",
    "queue_name": "Queue1",
    "vendor_ids": "vendor1,vendor2",
    "ctc_call_id": "call123",
    "state_elapsed_time": "00:05:30",
    "state_elapsed_seconds": 330,
    "call_duration_time": "00:02:15",
    "call_duration_seconds": 135,
    "last_event_time": "2025-11-19 07:15:00",
    "call_start_time": "2025-11-19 07:12:45"
  }
]
```

**PHP Example:**
```php
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5016/api/agents?account=ExampleAccount');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$agents = json_decode($response, true);
```

---

## CW App API (Port 5015)

### GET /api/queues
Retrieves queue data aggregated by path, including calls waiting and wait times.

**Request:**
```
GET http://172.20.60.110:5015/api/queues
```

**Response:**
```json
[
  {
    "path": "/queue/path1",
    "queue_name": "Queue1",
    "account_name": "Example Account",
    "calls_waiting": 5,
    "longest_wait": "001:23:45",
    "longest_wait_seconds": 5025,
    "agentgroup_presence": {
      "group1": 3,
      "group2": 1
    },
    "last_updated": "2025-11-19T07:15:00"
  }
]
```

**PHP Example:**
```php
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5015/api/queues');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$queues = json_decode($response, true);
```

### GET /api/agents
Retrieves agent status data.

**Request:**
```
GET http://172.20.60.110:5015/api/agents
```

**Response:**
```json
{
  "agent123": {
    "agent_id": "agent123",
    "agent_name": "John Doe",
    "function_event": "Agent Answer Remote ACD",
    "state_elapsed_time": "00:05:30",
    "state_elapsed_seconds": 330,
    "call_duration_time": "00:02:15",
    "call_duration_seconds": 135,
    "last_event_time": "2025-11-19 07:15:00",
    "queue_name": "Queue1",
    "ctc_call_id": "call123"
  }
}
```

**PHP Example:**
```php
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://172.20.60.110:5015/api/agents');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
$response = curl_exec($ch);
curl_close($ch);
$agents = json_decode($response, true);
```

---

## Notes
- All times are in HH:MM:SS format or ISO 8601 where applicable.
- Data is real-time and reflects the current state from Redis.
- For production use, consider implementing caching or rate limiting on the remote server to avoid overloading the APIs.
- Test connectivity using curl or a browser before integrating into PHP code.