-- Database Schema for Real-Time Dashboard
-- Compatible with MySQL 5.5

CREATE DATABASE IF NOT EXISTS realtime_dashboard;
USE realtime_dashboard;

-- Accounts table
CREATE TABLE accounts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    account_name VARCHAR(100) NOT NULL,
    account_code VARCHAR(50) NOT NULL UNIQUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8;

-- Agents table
CREATE TABLE agents (
    id INT AUTO_INCREMENT PRIMARY KEY,
    agent_id VARCHAR(50) NOT NULL UNIQUE,
    agent_name VARCHAR(100) NOT NULL,
    account_id INT NOT NULL,
    location VARCHAR(100),
    desk_location VARCHAR(50),
    status ENUM('Active', 'Idle', 'Not Ready') DEFAULT 'Idle',
    shift_duration TIME,
    inbound_calls INT DEFAULT 0,
    outbound_calls INT DEFAULT 0,
    inbound_avg_time INT DEFAULT 0,
    outbound_avg_time INT DEFAULT 0,
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (account_id) REFERENCES accounts(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8;

-- Queues table
CREATE TABLE queues (
    id INT AUTO_INCREMENT PRIMARY KEY,
    queue_name VARCHAR(100) NOT NULL,
    account_id INT NOT NULL,
    calls_in_queue INT DEFAULT 0,
    wait_time INT DEFAULT 0,
    status_indicator ENUM('red', 'yellow', 'green') DEFAULT 'green',
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (account_id) REFERENCES accounts(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8;

-- Account Metrics table
CREATE TABLE account_metrics (
    id INT AUTO_INCREMENT PRIMARY KEY,
    account_id INT NOT NULL,
    service_level DECIMAL(5,2) DEFAULT 0,
    service_level_target DECIMAL(5,2) DEFAULT 85.00,
    asa INT DEFAULT 0,
    asa_target INT DEFAULT 30,
    aht INT DEFAULT 0,
    aht_target INT DEFAULT 600,
    abandon_rate DECIMAL(5,2) DEFAULT 0,
    abandon_target DECIMAL(5,2) DEFAULT 5.00,
    agents_online INT DEFAULT 0,
    agents_scheduled INT DEFAULT 0,
    occupancy DECIMAL(5,2) DEFAULT 0,
    occupancy_target DECIMAL(5,2) DEFAULT 80.00,
    csat_score DECIMAL(3,2) DEFAULT 0,
    csat_target DECIMAL(3,2) DEFAULT 4.50,
    qa_compliance DECIMAL(5,2) DEFAULT 0,
    qa_target DECIMAL(5,2) DEFAULT 90.00,
    timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (account_id) REFERENCES accounts(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8;

-- Forecast Data table
CREATE TABLE forecast_data (
    id INT AUTO_INCREMENT PRIMARY KEY,
    account_id INT NOT NULL,
    time_slot TIME NOT NULL,
    forecast_calls INT DEFAULT 0,
    actual_calls INT DEFAULT 0,
    date_recorded DATE NOT NULL,
    FOREIGN KEY (account_id) REFERENCES accounts(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8;

-- SLA Trend Data table
CREATE TABLE sla_trend (
    id INT AUTO_INCREMENT PRIMARY KEY,
    account_id INT NOT NULL,
    time_slot TIME NOT NULL,
    sla_percentage DECIMAL(5,2) DEFAULT 0,
    asa_seconds INT DEFAULT 0,
    date_recorded DATE NOT NULL,
    FOREIGN KEY (account_id) REFERENCES accounts(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8;

-- Agent Login Data table
CREATE TABLE agent_login_data (
    id INT AUTO_INCREMENT PRIMARY KEY,
    account_id INT NOT NULL,
    time_slot TIME NOT NULL,
    logged_in INT DEFAULT 0,
    available INT DEFAULT 0,
    date_recorded DATE NOT NULL,
    FOREIGN KEY (account_id) REFERENCES accounts(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8;

-- Sample Data
INSERT INTO accounts (account_name, account_code) VALUES
('SNA', 'SNA'),
('Cloud5', 'CLOUD5'),
('DeepBlue', 'DEEPBLUE'),
('BPRF', 'BPRF'),
('SageNet', 'SAGENET'),
('Bright', 'BRIGHT');

-- Sample agents for SNA
INSERT INTO agents (agent_id, agent_name, account_id, location, desk_location, status, shift_duration, inbound_calls, outbound_calls, inbound_avg_time, outbound_avg_time) VALUES
('AG-1042', 'Sarah Mitchell', 1, 'Onsite', 'GenSan 1F', 'Active', '05:23:15', 16, 3, 510, 252),
('AG-1095', 'John Davis', 1, 'WFH', NULL, 'Active', '04:51:42', 22, 1, 468, 210),
('AG-1138', 'Maria Garcia', 1, 'Onsite', 'Cebu 9F', 'Idle', '06:12:08', 25, 5, 408, 312),
('AG-1156', 'Robert Chen', 1, 'WFH', NULL, 'Active', '03:38:52', 15, 2, 552, 378),
('AG-1189', 'Emily Johnson', 1, 'Onsite', 'GenSan 2F', 'Idle', '05:45:33', 28, 7, 372, 288),
('AG-1203', 'Michael Brown', 1, 'Onsite', 'Cebu 10F', 'Active', '07:15:20', 31, 4, 606, 432),
('AG-1247', 'Jennifer Lee', 1, 'WFH', NULL, 'Not Ready', '02:28:45', 12, 1, 528, 330),
('AG-1289', 'David Martinez', 1, 'Onsite', 'Cebu 4F', 'Idle', '06:03:18', 20, 6, 450, 294);

-- Sample queues for SNA
INSERT INTO queues (queue_name, account_id, calls_in_queue, wait_time, status_indicator) VALUES
('Tech Tier 2', 1, 19, 261, 'red'),
('Billing Support', 1, 12, 195, 'yellow'),
('General Inquiries', 1, 8, 128, 'green'),
('Tech Tier 1', 1, 15, 212, 'yellow'),
('Account Services', 1, 5, 83, 'green');

-- Sample account metrics
INSERT INTO account_metrics (account_id, service_level, asa, aht, abandon_rate, agents_online, agents_scheduled, occupancy, csat_score, qa_compliance) VALUES
(1, 87.4, 28, 552, 2.8, 48, 52, 82, 4.6, 91),
(2, 82.0, 45, 690, 4.8, 35, 42, 78, 4.3, 87),
(3, 91.0, 22, 528, 1.8, 62, 64, 85, 4.7, 93),
(4, 79.0, 52, 738, 5.5, 28, 38, 73, 4.1, 82),
(5, 89.0, 31, 576, 2.7, 44, 46, 84, 4.5, 90),
(6, 84.0, 38, 612, 3.2, 31, 34, 76, 4.4, 88);

-- Indexes for performance
CREATE INDEX idx_agents_status ON agents(status);
CREATE INDEX idx_agents_account ON agents(account_id);
CREATE INDEX idx_queues_account ON queues(account_id);
CREATE INDEX idx_metrics_account ON account_metrics(account_id);
CREATE INDEX idx_forecast_date ON forecast_data(date_recorded, account_id);
CREATE INDEX idx_sla_date ON sla_trend(date_recorded, account_id);
CREATE INDEX idx_login_date ON agent_login_data(date_recorded, account_id);