<?php
/**
 * Sample Data Generator for Real-Time Dashboard
 * Run this script to populate the database with realistic sample data
 */

require_once 'config/database.php';

$db = Database::getInstance()->getConnection();

echo "Generating sample data...\n\n";

// Generate forecast data for today
echo "Generating forecast data...\n";
$times = array('08:00', '08:30', '09:00', '09:30', '10:00');
$accounts = array(1, 2, 3, 4, 5, 6);

foreach ($accounts as $accountId) {
    foreach ($times as $time) {
        $forecast = rand(140, 195);
        $actual = rand(140, 190);
        
        $sql = "INSERT INTO forecast_data (account_id, time_slot, forecast_calls, actual_calls, date_recorded) 
                VALUES (:account_id, :time_slot, :forecast_calls, :actual_calls, CURDATE())
                ON DUPLICATE KEY UPDATE forecast_calls = VALUES(forecast_calls), actual_calls = VALUES(actual_calls)";
        
        $stmt = $db->prepare($sql);
        $stmt->execute(array(
            ':account_id' => $accountId,
            ':time_slot' => $time,
            ':forecast_calls' => $forecast,
            ':actual_calls' => $actual
        ));
    }
}

// Generate SLA trend data
echo "Generating SLA trend data...\n";
$slaTimes = array('08:00', '08:15', '08:30', '08:45', '09:00', '09:15', '09:30');

foreach ($accounts as $accountId) {
    foreach ($slaTimes as $time) {
        $sla = rand(80, 95);
        $asa = rand(20, 40);
        
        $sql = "INSERT INTO sla_trend (account_id, time_slot, sla_percentage, asa_seconds, date_recorded) 
                VALUES (:account_id, :time_slot, :sla_percentage, :asa_seconds, CURDATE())
                ON DUPLICATE KEY UPDATE sla_percentage = VALUES(sla_percentage), asa_seconds = VALUES(asa_seconds)";
        
        $stmt = $db->prepare($sql);
        $stmt->execute(array(
            ':account_id' => $accountId,
            ':time_slot' => $time,
            ':sla_percentage' => $sla,
            ':asa_seconds' => $asa
        ));
    }
}

// Generate agent login data
echo "Generating agent login data...\n";
$loginTimes = array('08:00', '08:30', '09:00', '09:30', '10:00');

foreach ($accounts as $accountId) {
    foreach ($loginTimes as $time) {
        $loggedIn = rand(40, 55);
        $available = rand(35, 50);
        
        $sql = "INSERT INTO agent_login_data (account_id, time_slot, logged_in, available, date_recorded) 
                VALUES (:account_id, :time_slot, :logged_in, :available, CURDATE())
                ON DUPLICATE KEY UPDATE logged_in = VALUES(logged_in), available = VALUES(available)";
        
        $stmt = $db->prepare($sql);
        $stmt->execute(array(
            ':account_id' => $accountId,
            ':time_slot' => $time,
            ':logged_in' => $loggedIn,
            ':available' => $available
        ));
    }
}

// Add more agents for other accounts
echo "Generating additional agents...\n";

$firstNames = array('John', 'Sarah', 'Michael', 'Emily', 'David', 'Jessica', 'Robert', 'Jennifer', 
                    'William', 'Lisa', 'James', 'Mary', 'Richard', 'Patricia', 'Thomas', 'Linda');
$lastNames = array('Smith', 'Johnson', 'Williams', 'Brown', 'Jones', 'Garcia', 'Miller', 'Davis',
                   'Rodriguez', 'Martinez', 'Hernandez', 'Lopez', 'Wilson', 'Anderson', 'Taylor');
$locations = array('Onsite', 'Onsite', 'WFH');
$desks = array('GenSan 1F', 'GenSan 2F', 'Cebu 4F', 'Cebu 9F', 'Cebu 10F', 'Manila 3F');
$statuses = array('Active', 'Active', 'Active', 'Idle', 'Not Ready');

// Accounts 2-6
for ($accountId = 2; $accountId <= 6; $accountId++) {
    $agentCount = rand(6, 10);
    
    for ($i = 0; $i < $agentCount; $i++) {
        $agentNum = 2000 + ($accountId * 100) + $i;
        $firstName = $firstNames[array_rand($firstNames)];
        $lastName = $lastNames[array_rand($lastNames)];
        $location = $locations[array_rand($locations)];
        $desk = $location === 'WFH' ? NULL : $desks[array_rand($desks)];
        $status = $statuses[array_rand($statuses)];
        
        $hours = rand(2, 8);
        $mins = rand(0, 59);
        $secs = rand(0, 59);
        $shiftDuration = sprintf('%02d:%02d:%02d', $hours, $mins, $secs);
        
        $inboundCalls = rand(10, 35);
        $outboundCalls = rand(1, 8);
        $inboundAvg = rand(300, 700);
        $outboundAvg = rand(180, 450);
        
        $agentId = 'AG-' . $agentNum;
        $agentName = $firstName . ' ' . $lastName;
        
        // Check if agent exists
        $checkSql = "SELECT id FROM agents WHERE agent_id = :agent_id";
        $checkStmt = $db->prepare($checkSql);
        $checkStmt->execute(array(':agent_id' => $agentId));
        
        if ($checkStmt->rowCount() > 0) {
            // Update existing agent
            $sql = "UPDATE agents SET 
                    status = :status,
                    shift_duration = :shift_duration,
                    inbound_calls = :inbound_calls,
                    outbound_calls = :outbound_calls,
                    inbound_avg_time = :inbound_avg_time,
                    outbound_avg_time = :outbound_avg_time
                    WHERE agent_id = :agent_id";
            
            $stmt = $db->prepare($sql);
            $stmt->execute(array(
                ':agent_id' => $agentId,
                ':status' => $status,
                ':shift_duration' => $shiftDuration,
                ':inbound_calls' => $inboundCalls,
                ':outbound_calls' => $outboundCalls,
                ':inbound_avg_time' => $inboundAvg,
                ':outbound_avg_time' => $outboundAvg
            ));
        } else {
            // Insert new agent
            $sql = "INSERT INTO agents (agent_id, agent_name, account_id, location, desk_location, status, 
                    shift_duration, inbound_calls, outbound_calls, inbound_avg_time, outbound_avg_time)
                    VALUES (:agent_id, :agent_name, :account_id, :location, :desk_location, :status,
                    :shift_duration, :inbound_calls, :outbound_calls, :inbound_avg_time, :outbound_avg_time)";
            
            $stmt = $db->prepare($sql);
            $stmt->execute(array(
                ':agent_id' => $agentId,
                ':agent_name' => $agentName,
                ':account_id' => $accountId,
                ':location' => $location,
                ':desk_location' => $desk,
                ':status' => $status,
                ':shift_duration' => $shiftDuration,
                ':inbound_calls' => $inboundCalls,
                ':outbound_calls' => $outboundCalls,
                ':inbound_avg_time' => $inboundAvg,
                ':outbound_avg_time' => $outboundAvg
            ));
        }
    }
}

// Add queues for other accounts
echo "Generating queues for all accounts...\n";

$queueTemplates = array(
    array('name' => 'Tech Support Tier 1', 'calls' => array(12, 18), 'wait' => array(150, 250)),
    array('name' => 'Tech Support Tier 2', 'calls' => array(15, 25), 'wait' => array(200, 300)),
    array('name' => 'Billing Support', 'calls' => array(10, 16), 'wait' => array(120, 220)),
    array('name' => 'General Inquiries', 'calls' => array(5, 12), 'wait' => array(80, 150)),
    array('name' => 'Account Services', 'calls' => array(3, 8), 'wait' => array(60, 120)),
    array('name' => 'Sales Support', 'calls' => array(8, 15), 'wait' => array(100, 200))
);

for ($accountId = 2; $accountId <= 6; $accountId++) {
    $queueCount = rand(4, 6);
    
    for ($i = 0; $i < $queueCount; $i++) {
        $template = $queueTemplates[$i];
        $calls = rand($template['calls'][0], $template['calls'][1]);
        $wait = rand($template['wait'][0], $template['wait'][1]);
        $queueName = $template['name'];
        
        // Determine status based on wait time
        if ($wait > 240) $statusIndicator = 'red';
        else if ($wait > 180) $statusIndicator = 'yellow';
        else $statusIndicator = 'green';
        
        // Check if queue exists
        $checkSql = "SELECT id FROM queues WHERE queue_name = :queue_name AND account_id = :account_id";
        $checkStmt = $db->prepare($checkSql);
        $checkStmt->execute(array(
            ':queue_name' => $queueName,
            ':account_id' => $accountId
        ));
        
        if ($checkStmt->rowCount() > 0) {
            // Update existing queue
            $sql = "UPDATE queues SET 
                    calls_in_queue = :calls_in_queue,
                    wait_time = :wait_time,
                    status_indicator = :status_indicator
                    WHERE queue_name = :queue_name AND account_id = :account_id";
            
            $stmt = $db->prepare($sql);
            $stmt->execute(array(
                ':queue_name' => $queueName,
                ':account_id' => $accountId,
                ':calls_in_queue' => $calls,
                ':wait_time' => $wait,
                ':status_indicator' => $statusIndicator
            ));
        } else {
            // Insert new queue
            $sql = "INSERT INTO queues (queue_name, account_id, calls_in_queue, wait_time, status_indicator)
                    VALUES (:queue_name, :account_id, :calls_in_queue, :wait_time, :status_indicator)";
            
            $stmt = $db->prepare($sql);
            $stmt->execute(array(
                ':queue_name' => $queueName,
                ':account_id' => $accountId,
                ':calls_in_queue' => $calls,
                ':wait_time' => $wait,
                ':status_indicator' => $statusIndicator
            ));
        }
    }
}

// Update metrics with some variation
echo "Updating account metrics...\n";

$sql = "SELECT id, service_level, asa, aht, abandon_rate, occupancy FROM account_metrics";
$stmt = $db->prepare($sql);
$stmt->execute();
$metrics = $stmt->fetchAll();

foreach ($metrics as $metric) {
    $newSLA = $metric['service_level'] + (rand(-200, 400) / 100);
    $newASA = $metric['asa'] + rand(-5, 5);
    $newAHT = $metric['aht'] + rand(-30, 30);
    $newAbandon = $metric['abandon_rate'] + (rand(-50, 100) / 100);
    $newOccupancy = $metric['occupancy'] + (rand(-200, 400) / 100);
    
    // Keep values in reasonable ranges
    $newSLA = max(70, min(95, $newSLA));
    $newASA = max(20, min(60, $newASA));
    $newAHT = max(300, min(900, $newAHT));
    $newAbandon = max(1, min(8, $newAbandon));
    $newOccupancy = max(70, min(90, $newOccupancy));
    
    $updateSql = "UPDATE account_metrics SET 
        service_level = :service_level,
        asa = :asa,
        aht = :aht,
        abandon_rate = :abandon_rate,
        occupancy = :occupancy,
        timestamp = NOW()
        WHERE id = :id";
    
    $updateStmt = $db->prepare($updateSql);
    $updateStmt->execute(array(
        ':id' => $metric['id'],
        ':service_level' => round($newSLA, 1),
        ':asa' => intval($newASA),
        ':aht' => intval($newAHT),
        ':abandon_rate' => round($newAbandon, 1),
        ':occupancy' => round($newOccupancy, 0)
    ));
}

echo "\n✓ Sample data generation complete!\n";
echo "You can now view the dashboard in your browser.\n";
echo "To simulate real-time updates, run this script periodically or set up a cron job.\n";

?>